#!/bin/bash
###############################################################################
#
#          Dell Inc. PROPRIETARY INFORMATION
#
#  This software is supplied under the terms of a license agreement or
#  nondisclosure agreement with Dell Inc. and may not
#  be copied or disclosed except in accordance with the terms of that
#  agreement.
#
#  Copyright (c) 2000-2019 Dell Inc. All Rights Reserved.
#
#
#  Revision:
#    $Revision: $
#
#  Last Modified By / On:
#    $Author: $ / $Date: $
#
#  Author:
#    OpenManage Install
#
#  Environment:
#    Linux
#
#  User/Kernel Mode:
#    User
#
#  Abstract/Purpose:
#    Functions used by OM components to decide the type of system.
#
#  See Also:
#
#
#  Notes:
#
#
###############################################################################
TRUE=0
FALSE=1
NON_DELL=1
LOWER_GENERATION=2
CLIENT_SYSTEM=3
SUPPORTED_DELL_SERVER=0

CLIENT_SYSTEM_LIST=prereq/ClientSystemList.txt

# Check whether this utility is executed by install scripts.
# If so, CLIENT_SYSTEM_LIST have to be picked from a
# different location
if [ -n "$1" ] && [ -n "$2" ] && [ -d "$2" ]; then
   if [ "$1" == "dvd" ] || [ "$1" == "DVD" ]; then
         # Getting executed from install script.
         SUPPORT_DIR=`echo "$2" | sed "s/ /\\\ /"`
         CLIENT_SYSTEM_LIST="$SUPPORT_DIR/ClientSystemList.txt"
   fi
fi
if [ -n "$1" ]
then
         SUPPORT_DIR=`echo "$1" | sed "s/ /\\\ /"`
         CLIENT_SYSTEM_LIST="$SUPPORT_DIR/prereq/ClientSystemList.txt"
fi
#############
## CheckForOverride()
## Function to check whether override is enabled.
## If enabled, then return 0.
#############
CheckForOverride() 
{
   [ -n ${NO_SYSCHECK} ] || return $TRUE
   [ "${IGNORE_GENERATION}" != "1" ] || return $TRUE
   return $FALSE
}

#############
## IsDELLSystem()
## Function to check whether it is a DELL system and then return 0.
## It can be a server or a client system
#############
IsDELLSystem() 
{
   dmidecode | grep "String 1" | awk -F":" '{print $2}' | grep -qi "Dell"
   if [ "$?" = "0" ]; then
       return $TRUE
   else
       return $FALSE
   fi
}

#############
## IsHigherGeneration()
## Compares the system ID with the range of 15G sysid
## If higher than 0x08FC and less than 0x0900, then return 0
#############
IsHigherGeneration() 
{
    if [ -n "${OM_SYSTEM_ID}" ]
    then
       SYSID_HEX="0x${OM_SYSTEM_ID}"
    else
    SYSID_STR=`dmidecode -t 208 | tail -3 | sed '2q;d'`
	SYSID_STR=`echo ${SYSID_STR:28:3}${SYSID_STR:25:3}`
	SYSID_HEX=`echo 0x${SYSID_STR//[[:blank:]]/}`
    fi

    SYSID_DEC=`printf "%d"  $SYSID_HEX`    
    MIN_SUPPORTED_SYSID_HEX=0x048C
    MIN_SUPPORTED_SYSID_DEC=`printf "%d" $MIN_SUPPORTED_SYSID_HEX`

    if [ $SYSID_DEC -ge $MIN_SUPPORTED_SYSID_DEC ]; then
       # Supported Generation.
       return $TRUE
    else
       return $FALSE
    fi
}

#############
## IsClientSystem()
## Compares the product name with all client systems
## If any matches, then it is a client system and return TRUE.
#############
IsClientSystem()
{
# Since we have multiple Product names in different generation systems. We are checking
# DMI type 1 to fetch the proper product name.
    PRODUCT_NAME=`dmidecode -t 1 | grep  "Product Name" | awk -F":" '{print $2}'`

    if [ -n "${PRODUCT_NAME}" ] && [ -f "$CLIENT_SYSTEM_LIST" ]; then
		while read client_name
        do
            echo "$PRODUCT_NAME" | grep -qi "$client_name"
            if [ "$?" = "0" ]; then
                CLIENT_SYSTEM=0
                return $TRUE
            fi
        done < "$CLIENT_SYSTEM_LIST"


        # Not a client system
        return $FALSE
    fi

    #Not able to find the product name
    #Assume it is a client system.
    return $TRUE
}

#############
## IsThisSupportedGeneration()
## Function to check whether it is a DELL server of supported generation.
## If yes, then return 0; else return 1.
#############
IsThisSupportedGeneration()
{
    CheckForOverride && return $SUPPORTED_DELL_SERVER

    IsDELLSystem || return $NON_DELL

    IsHigherGeneration || return $LOWER_GENERATION

    [ -n "${OM_SYSTEM_ID}" ] && return $SUPPORTED_DELL_SERVER   

    IsClientSystem && return $CLIENT_SYSTEM

    return $SUPPORTED_DELL_SERVER
}
