#!/bin/bash
#-------------------------------------------------------------------------
#
#          DELL COMPUTER CORPORATION PROPRIETARY INFORMATION
#
#  This software is supplied under the terms of a license agreement or
#  nondisclosure agreement with Dell Computer Corporation and may not
#  be copied or disclosed except in accordance with the terms of that
#  agreement.
#
#  Copyright (c) 2021 Dell Computer Corp. All Rights Reserved.
#
#  Abstract/Purpose:
#  To execute the commands used by iDRAC Service Module
#------------------------------------------------------------------------

DcismExternalUtilUsage()
{
	echo "This script is used internally by iDRAC Service Module"
	echo "Exiting..."
}


#this isused by DCOSFPI when netplan DHCP mechanism is used in Ubuntu servers
#using the netplan command to create temp lease file
OSINFOCreateNetplanLease()
{
	InterfaceName=$1
	LeaseFileName=$2
	umask 0077 && /usr/sbin/netplan ip leases $InterfaceName > $LeaseFileName
	ConnectionName=`nmcli device show $InterfaceName | grep -w "GENERAL.CONNECTION" | sed 's/GENERAL.CONNECTION://' | awk '{$1=$1};1'`
	nmcli con show $ConnectionName | grep -w "IP6.GATEWAY" | sed 's/:[[:blank:]]*/: /' >> $LeaseFileName
    nmcli con show $ConnectionName | grep -w "ipv6.dns:" >> $LeaseFileName
	return $?
}

#this isused by DCOSFPI when internal DHCP mechanism is used in RHEL servers
#using the nmcli con show <interface name>  command output to create temp lease file for internal mode
OSINFOCreateInternalLease()
{
	InterfaceName=$1
	LeaseFileName=$2
	umask 0077 && touch $LeaseFileName
	cd /var/lib/NetworkManager/
	cat *$InterfaceName.lease | grep -w SERVER_ADDRESS >>$LeaseFileName
	nmcli con show $InterfaceName | grep -w ip_address >> $LeaseFileName
	nmcli con show $InterfaceName | grep -w routers >> $LeaseFileName
	nmcli con show $InterfaceName | grep -w dhcp_server_identifier >> $LeaseFileName
	nmcli con show $InterfaceName | grep -w domain_name_servers >> $LeaseFileName
	nmcli con show $InterfaceName | grep -w "IP6.GATEWAY" | sed 's/:[[:blank:]]*/: /' >> $LeaseFileName
	nmcli con show $InterfaceName | grep -w "ipv6.dns:" >> $LeaseFileName
	return 0
}

#This is used by ISM SPD logs
SPDLOGSCheckProcessorType()
{
	grep GenuineIntel /proc/cpuinfo > /dev/null 2>&1
	return $?
}

#This is used by ISM feature snmpget
SNMPGETFPIEnableSNMPProxyConfigCmd()
{
	#echo "Inside SNMPGETFPIEnableSNMPProxyConfigCmd"
	SnmpCommunity=$1
	IDRACUserPassword=$2
	UsbNicIPAddr=$3
	SnmpAgentPort=$4
	IdracRootOID=$5
	echo "proxy -c $SnmpCommunity -v3 -u iSMSnmpUser -a SHA -A $IDRACUserPassword -x AES -X $IDRACUserPassword -l authPriv $UsbNicIPAddr:$SnmpAgentPort $IdracRootOID $IdracRootOID" >> /etc/snmp/snmpd.conf
	service snmpd stop > /dev/null 2>&1
	service snmpd start > /dev/null 2>&1
	
	return 0
}

#This is used by ISM feature snmpget
SNMPGETFPIDisableSNMPProxyConfigCmd()
{
	#echo "Inside DisableSNMPProxyConfigCmd"
	SnmpCommunity=$1
	IDRACUserPassword=$2
	UsbNicIPAddr=$3
	SnmpAgentPort=$4
	IdracRootOID=$5
	cp -p /etc/snmp/snmpd.conf /opt/dell/srvadmin/iSM/etc/snmpd_backup.conf
	cat /etc/snmp/snmpd.conf | grep -v "proxy -c $SnmpCommunity -v3 -u iSMSnmpUser -a SHA -A $IDRACUserPassword -x AES -X $IDRACUserPassword -l authPriv $UsbNicIPAddr:$SnmpAgentPort $IdracRootOID $IdracRootOID" > /opt/dell/srvadmin/iSM/etc/snmpd_backup.conf
	mv /opt/dell/srvadmin/iSM/etc/snmpd_backup.conf /etc/snmp/snmpd.conf
	service snmpd stop > /dev/null 2>&1
	service snmpd start > /dev/null 2>&1
	
	return 0
}

#This is used by ISM feature os2idrac
OS2IDRACFPIRemoveLocalHostRulesCmd()
{
	ListenPortNumReg=$1
	ConnectAddress=$2
	ConnectPortReg=$3
	iptables -t nat -C OUTPUT -m addrtype --src-type LOCAL --dst-type LOCAL -p tcp --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
	if [ $? -eq 0 ]
	then 
		iptables -t nat -D OUTPUT -m addrtype --src-type LOCAL --dst-type LOCAL -p tcp --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
		if [ $? -ne 0 ]
		then 
			return 12
		fi
	fi	
	return 0
}

#This is used by ISM feature os2idrac
OS2IDRACFPIAddLocalHostRulesCmd()
{
	ListenPort=$1
	ConnectAddr=$2
	ConnectPort=$3
	
	sysctl -w net.ipv4.conf.all.route_localnet=1
	if [ $? -ne 0 ]
	then 
		return 9
	fi
	
	iptables -t nat -C OUTPUT -m addrtype --src-type LOCAL --dst-type LOCAL -p tcp --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 1>&2
	if [ $? -ne 0 ]
	then
			iptables -t nat -A OUTPUT -m addrtype --src-type LOCAL --dst-type LOCAL -p tcp --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 1>&2
		if [ $? -ne 0 ]
		then 
			return 10
		fi
	fi	
	return 0
}

#This is used by ISM feature os2idrac
OS2IDRACFPIDisableOS2iDRACConfigCmd()
{
	ListenPortNumReg=$2
	ConnectAddress=$3
	ConnectPortReg=$4
	LocalIdracIP=$5
	if [ "$1" == "0" ]
	then
		iptables -t nat -C PREROUTING -p tcp --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
		if [ $? -eq 0 ]
		then
			iptables -t nat -D PREROUTING -p tcp --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
			if [ $? -ne 0 ]
			then 
				return 7
			fi
		fi
	else
		RangeAddress=$6
		iptables -t nat -C PREROUTING -p tcp -s $RangeAddress --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
		if [ $? -eq 0 ]
		then
			iptables -t nat -D PREROUTING -p tcp -s $RangeAddress --dport $ListenPortNumReg -j DNAT --to-destination $ConnectAddress:$ConnectPortReg 1>&2
			if [ $? -ne 0 ]
			then 
				return 7
			fi
		fi
	fi
	
	iptables -t nat -C POSTROUTING -p tcp -d $ConnectAddress --dport $ConnectPortReg -j SNAT --to-source $LocalIdracIP 1>&2
	if [ $? -eq 0 ]
	then
		iptables -t nat -D POSTROUTING -p tcp -d $ConnectAddress --dport $ConnectPortReg -j SNAT --to-source $LocalIdracIP 1>&2
		if [ $? -ne 0 ]
		then
			return 8
		fi
	fi
	
	iptables -L OS2iDRAC 1>&2
	if [ $? -eq 0 ]
	then
		iptables -F OS2iDRAC
		if [ $? -ne 0 ]
		then
			return 9
		fi
	fi
	
	iptables -C FORWARD -j OS2iDRAC 1>&2
	if [ $? -eq 0 ]
	then
		iptables -D FORWARD -j OS2iDRAC
		if [ $? -ne 0 ]
		then
			return 10
		fi
	fi
	
	iptables -L OS2iDRAC   1>&2
	if [ $? -eq 0 ]
	then
		iptables -X OS2iDRAC
		if [ $? -ne 0 ]
		then
			return 11
		fi
	fi
	
	return 0
}

#This is used by ISM feature os2idrac
OS2IDRACFPIEnableOS2iDRACConfigCmd1()
{
	echo 1 > /proc/sys/net/ipv4/ip_forward
	
	return 0
}


#This is used by ISM feature os2idrac
OS2IDRACFPIEnableOS2iDRACConfigCmd2()
{
	ListenPort=$2
	ConnectAddr=$3
	ConnectPort=$4
	LocalIdracIP=$5
	IdracInterfaceName=$6
	
	if [ "$1" == "0" ]
	then
		iptables -t nat -C PREROUTING -p tcp --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 2>/dev/null
		if [ $? -ne 0 ]
		then
			iptables -t nat -A PREROUTING -p tcp --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 2>/dev/null
			if [ $? -ne 0 ]
			then
				return 1
			fi
		fi
	else
		RangeIP=$7
		iptables -t nat -C PREROUTING -p tcp -s $RangeIP --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 2>/dev/null
		if [ $? -ne 0 ]
		then
			iptables -t nat -A PREROUTING -p tcp -s $RangeIP --dport $ListenPort -j DNAT --to-destination $ConnectAddr:$ConnectPort 2>/dev/null
			if [ $? -ne 0 ]
			then
				return 1
			fi
		fi
	fi
	
	iptables -t nat -C POSTROUTING -p tcp -d $ConnectAddr --dport $ConnectPort -j SNAT --to-source $LocalIdracIP 2>/dev/null
	if [ $? -ne 0 ]
	then
		iptables -t nat -A POSTROUTING -p tcp -d $ConnectAddr --dport $ConnectPort -j SNAT --to-source $LocalIdracIP 2>/dev/null
		if [ $? -ne 0 ]
		then
			return 2
		fi
	fi
	
	iptables -L OS2iDRAC  2>/dev/null
	if [ $? -ne 0 ]
	then
		iptables -N OS2iDRAC 2>/dev/null
		if [ $? -ne 0 ]
		then
			return 3
		fi
	fi
	
	iptables -C FORWARD -j OS2iDRAC 2>/dev/null
	if [ $? -ne 0 ]
	then
		iptables -I FORWARD -j OS2iDRAC 2>/dev/null
		if [ $? -ne 0 ]
		then
			return 4
		fi
	fi
	
	iptables -C OS2iDRAC -p tcp -d $ConnectAddr --dport $ConnectPort -j ACCEPT 2>/dev/null
	if [ $? -ne 0 ]
	then
		iptables -A OS2iDRAC -p tcp -d $ConnectAddr --dport $ConnectPort -j ACCEPT 2>/dev/null
		if [ $? -ne 0 ]
		then
			return 5
		fi
	fi
	
	iptables -C OS2iDRAC -p tcp -i $IdracInterfaceName -s $ConnectAddr -j ACCEPT 2>/dev/null
	if [ $? -ne 0 ]
	then
		iptables -A OS2iDRAC -p tcp -i $IdracInterfaceName -s $ConnectAddr -j ACCEPT 2>/dev/null
		if [ $? -ne 0 ]
		then
			return 6
		fi
	fi
	
	return 0
}

AddInbandFirewallRule()
{
	ConnectAddr=$1
	ConnectPort=$2
	#Add if the rule not exists
	systemctl -q is-active firewalld 2>/dev/null
	if [ $? -eq 0 ]
	then
		firewall-cmd --permanent --direct --add-rule ipv4 filter OUTPUT 0 -d $ConnectAddr -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
		firewall-cmd --reload 2>/dev/null
	else
		iptables -C OUTPUT -d $ConnectAddr -p tcp -m tcp --dport $ConnectPort -j ACCEPT || iptables -A OUTPUT -d $ConnectAddr/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
	fi
	return 0
}

RemoveInbandFirewallRule()
{

	ConnectAddr=$1
	ConnectPort=$2
	#remove if exist
	systemctl -q is-active firewalld 2>/dev/null
	if [ $? -eq 0 ]
	then
		firewall-cmd --permanent --direct --remove-rule ipv4 filter OUTPUT 0 -d $ConnectAddr -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
		firewall-cmd --reload 2>/dev/null
	else
		iptables -C OUTPUT -d $ConnectAddr -p tcp -m tcp --dport $ConnectPort -j ACCEPT && iptables -D OUTPUT -d $ConnectAddr/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
	fi
	return 0
}

AddInbandFirewallRuleIPV6()
{
	ConnectAddr=$1
	ConnectPort=$2
	#Add if the rule not exists
	systemctl -q is-active firewalld 2>/dev/null
	if [ $? -eq 0 ]
	then
		firewall-cmd --permanent --direct --add-rule ipv6 filter OUTPUT 0 -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
		firewall-cmd --reload 2>/dev/null
	else
		ip6tables  -C OUTPUT -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT || ip6tables -A OUTPUT -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
	fi
	return 0
}


RemoveInbandFirewallRuleIPV6()
{

	ConnectAddr=$1
	ConnectPort=$2
	#remove if exist
	systemctl -q is-active firewalld 2>/dev/null
	if [ $? -eq 0 ]
	then
		firewall-cmd --permanent --direct --remove-rule ipv6 filter OUTPUT 0 -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
		firewall-cmd --reload 2>/dev/null
	else
		ip6tables -C OUTPUT -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT && ip6tables -D OUTPUT -d fe80::1234/32 -p tcp -m tcp --dport $ConnectPort -j ACCEPT 2>/dev/null
	fi
	return 0
}


if [ $# -eq 0 ]
then
	DcismExternalUtilUsage
	exit 1
fi
command="$1"

case "$command" in
"EnableSNMPProxyConfig")
	if [ $# -lt 6 ]
	then
		exit 1
	fi
	SNMPGETFPIEnableSNMPProxyConfigCmd $2 $3 $4 $5 $6
	exit $?
	;;
	
"DisableSNMPProxyConfig")
	if [ $# -lt 6 ]
	then
		exit 1
	fi
	SNMPGETFPIDisableSNMPProxyConfigCmd $2 $3 $4 $5 $6
	exit $?
	;;
	
"RemoveLocalHostRules")
	if [ $# -lt 4 ]
	then
		exit 1
	fi
	OS2IDRACFPIRemoveLocalHostRulesCmd $2 $3 $4
	exit 0
	;;
	
"AddLocalHostRules")
	if [ $# -lt 4 ]
	then
		exit 1
	fi
	OS2IDRACFPIAddLocalHostRulesCmd $2 $3 $4
	exit 0
	;;

"DisableOS2iDRACConfig")
	if [ "$2" == "0" ]
	then
		if [ $# -lt 6 ]
		then
			exit 1
		fi
		OS2IDRACFPIDisableOS2iDRACConfigCmd $2 $3 $4 $5 $6
		exit 0
	else
		if [ $# -lt 7 ]
		then
			exit 1
		fi
		OS2IDRACFPIDisableOS2iDRACConfigCmd $2 $3 $4 $5 $6 $7
		exit 0
	fi
	;;
	
"EnableOS2iDRACConfig1")
	OS2IDRACFPIEnableOS2iDRACConfigCmd1
	exit 0
	;;

"EnableOS2iDRACConfig2")
	if [ "$2" == "0" ]
	then
		if [ $# -lt 7 ]
		then
			exit 1
		fi
		OS2IDRACFPIEnableOS2iDRACConfigCmd2 $2 $3 $4 $5 $6 $7
		exit 0
	else
		if [ $# -lt 8 ]
		then
			exit 1
		fi
		OS2IDRACFPIEnableOS2iDRACConfigCmd2 $2 $3 $4 $5 $6 $7 $8
		exit 0
	fi
	;;

"CheckProcessorType")
	SPDLOGSCheckProcessorType
	exit $?
	;;

"CreateNetplanLease")
	if [ $# -lt 3 ]
	then
		exit 1
	fi
	OSINFOCreateNetplanLease $2 $3
	exit $?
	;;
#script is called for this case for RHEL8.x in DHCP internal mode
"CreateInternalLease")
	if [ $# -lt 3 ]
	then
		exit 1
	fi
	OSINFOCreateInternalLease $2 $3
	exit $?
	;;
"EnableISMFirewallRule")
	if [ $# -lt 3 ]
	then
		exit 1
	fi
	AddInbandFirewallRule $2 $3
	AddInbandFirewallRuleIPV6 $2 $3
	exit $?
	;;
"DisableISMFirewallRule")
	if [ $# -lt 3 ]
	then
		exit 1
	fi
	RemoveInbandFirewallRule $2 $3
	RemoveInbandFirewallRuleIPV6 $2 $3
	exit $?
	;;
*)
	#echo "Invalid Input passed for the ISM_commands script"
	;;
esac

exit 0



